# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# <pep8 compliant>


import bpy
from bpy.props import (
    StringProperty,
    EnumProperty,
)

from .ops import show_mouse_hold_status
from .ui import SK_PT_ScreencastKeys, SK_PT_ScreencastKeys_Overlay
from .utils.addon_updater import AddonUpdaterManager
from .utils.bl_class_registry import BlClassRegistry
from .utils import compatibility as compat
from . import common


@BlClassRegistry()
class SK_OT_CheckAddonUpdate(bpy.types.Operator):
    bl_idname = "wm.sk_check_addon_update"
    bl_label = "Check Update"
    bl_description = "Check Add-on Update"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        updater = AddonUpdaterManager.get_instance()
        updater.check_update_candidate()

        return {'FINISHED'}


@BlClassRegistry()
@compat.make_annotations
class SK_OT_UpdateAddon(bpy.types.Operator):
    bl_idname = "wm.sk_update_addon"
    bl_label = "Update"
    bl_description = "Update Add-on"
    bl_options = {'REGISTER', 'UNDO'}

    branch_name = StringProperty(
        name="分枝名称",
        description="要更新的分支名称",
        default="",
    )

    def execute(self, context):
        updater = AddonUpdaterManager.get_instance()
        updater.update(self.branch_name)

        return {'FINISHED'}


@BlClassRegistry()
@compat.make_annotations
class SK_OT_SelectCustomMouseImage(bpy.types.Operator):
    bl_idname = "wm.sk_select_custom_mouse_image"
    bl_label = "Select Custom Mouse Image"
    bl_description = "Select custom mouse image"
    bl_options = {'REGISTER', 'UNDO'}

    target = bpy.props.EnumProperty(
        name="目标",
        description="打开图像文件的目标",
        items=[
            ('BASE', "Base", "Base image for custom mouse image"),
            ('OVERLAY_LEFT_MOUSE', "Overlay Left Mouse", "Overlay left mouse for custom mouse image"),
            ('OVERLAY_RIGHT_MOUSE', "Overlay Right Mouse", "Overlay right mouse for custom mouse image"),
            ('OVERLAY_MIDDLE_MOUSE', "Overlay Middle Mouse", "Overlay middle mouse for custom mouse image"),
        ],
        default='BASE',
    )

    filepath = bpy.props.StringProperty(
        subtype="FILE_PATH"
    )

    def invoke(self, context, event):
        wm = context.window_manager
        wm.fileselect_add(self)

        return {'RUNNING_MODAL'}

    def execute(self, context):
        prefs = compat.get_user_preferences(context).addons[__package__].preferences

        if self.target == 'BASE':
            prefs.custom_mouse_image_base = self.filepath
        elif self.target == 'OVERLAY_LEFT_MOUSE':
            prefs.custom_mouse_image_overlay_left_mouse = self.filepath
        elif self.target == 'OVERLAY_RIGHT_MOUSE':
            prefs.custom_mouse_image_overlay_right_mouse = self.filepath
        elif self.target == 'OVERLAY_MIDDLE_MOUSE':
            prefs.custom_mouse_image_overlay_middle_mouse = self.filepath

        return {'FINISHED'}


def get_update_candidate_branches(_, __):
    updater = AddonUpdaterManager.get_instance()
    if not updater.candidate_checked():
        return []

    return [(name, name, "") for name in updater.get_candidate_branch_names()]


@compat.make_annotations
class DisplayEventTextAliasProperties(bpy.types.PropertyGroup):
    alias_text = bpy.props.StringProperty(name="别名文本", default="")
    default_text = bpy.props.StringProperty(options={'HIDDEN'})
    event_id = bpy.props.StringProperty(options={'HIDDEN'})


def remove_custom_mouse_image(prefs, context):
    def remove_image(image_name):
        if image_name in bpy.data.images:
            image = bpy.data.images[image_name]
            bpy.data.images.remove(image)

    remove_image(common.CUSTOM_MOUSE_IMAGE_BASE_NAME)
    remove_image(common.CUSTOM_MOUSE_IMAGE_OVERLAY_LEFT_MOUSE_NAME)
    remove_image(common.CUSTOM_MOUSE_IMAGE_OVERLAY_RIGHT_MOUSE_NAME)
    remove_image(common.CUSTOM_MOUSE_IMAGE_OVERLAY_MIDDLE_MOUSE_NAME)


def update_custom_mouse_size(self, context):
    if ("use_custom_mouse_image_size" in self) and self["use_custom_mouse_image_size"]:
        if common.CUSTOM_MOUSE_IMAGE_BASE_NAME in bpy.data.images:
            image = bpy.data.images[common.CUSTOM_MOUSE_IMAGE_BASE_NAME]
            self["custom_mouse_size"] = image.size


@BlClassRegistry()
@compat.make_annotations
class SK_Preferences(bpy.types.AddonPreferences):
    bl_idname = __package__

    category = EnumProperty(
        name="类别",
        description="偏好类别",
        items=[
            ('CONFIG', "配置", "关于这个插件的配置"),
            ('DISPLAY_EVENT_TEXT_ALIAS', "显示事件文字别名", "用于显示的事件文字别名"),
            ('UPDATE', "更新", "更新此插件"),
        ],
        default='CONFIG'
    )

    # for Config.
    color = bpy.props.FloatVectorProperty(
        name="颜色",
        default=(1.0, 1.0, 1.0),
        min=0.0,
        max=1.0,
        subtype='COLOR_GAMMA',
        size=3
    )

    shadow = bpy.props.BoolProperty(
        name="阴影",
        default=False
    )

    shadow_color = bpy.props.FloatVectorProperty(
        name="阴影颜色",
        default=(0.0, 0.0, 0.0),
        min=0.0,
        max=1.0,
        subtype='COLOR',
        size=3
    )

    background = bpy.props.BoolProperty(
        name="背景",
        default=False
    )

    background_mode = bpy.props.EnumProperty(
        name="背景模式",
        items=[
            ('TEXT', "Text", ""),
            ('DRAW_AREA', "Draw Area", ""),
        ],
        default='DRAW_AREA',
    )

    background_color = bpy.props.FloatVectorProperty(
        name="背景颜色",
        default=(0.0, 0.0, 0.0),
        min=0.0,
        max=1.0,
        subtype='COLOR',
        size=3
    )

    background_rounded_corner_radius = bpy.props.IntProperty(
        name="背景圆角半径",
        description="背景的圆角半径",
        default=0,
        min=0,
        max=100,
    )

    font_size = bpy.props.IntProperty(
        name="字体大小",
        default=int(compat.get_user_preferences(bpy.context).ui_styles[0].widget.points),
        min=6,
        max=1000
    )

    margin = bpy.props.IntProperty(
        name="边距",
        description="边距",
        default=0,
        min=0,
        max=1000
    )

    line_thickness = bpy.props.FloatProperty(
        name="线条厚度",
        default=1,
        min=1,
        max=100
    )

    mouse_size = bpy.props.IntProperty(
        name="鼠标大小",
        default=int(compat.get_user_preferences(bpy.context).ui_styles[0].widget.points*3),
        min=18,
        max=1000,
    )

    origin = bpy.props.EnumProperty(
        name="起源",
        items=[
            ('REGION', "Region", ""),
            ('AREA', "Area", ""),
            ('WINDOW', "Window", ""),
            ('CURSOR', "Cursor", ""),
        ],
        default='REGION',
    )

    offset = bpy.props.IntVectorProperty(
        name="偏移",
        default=(20, 80),
        size=2,
        subtype='XYZ',
    )

    align = bpy.props.EnumProperty(
        name="对齐",
        items=[
            ('LEFT', "Left", ""),
            ('CENTER', "Center", ""),
            ('RIGHT', "Right", ""),
        ],
        default='LEFT'
    )

    display_time = bpy.props.FloatProperty(
        name="显示时间",
        default=3.0,
        min=0.5,
        max=10.0,
        step=10,
        subtype='TIME'
    )

    max_event_history = bpy.props.IntProperty(
        name="最大事件历史",
        description="要显示的最大事件历史记录数",
        default=5,
        min=1,
        step=1,
    )

    repeat_count = bpy.props.BoolProperty(
        name="重复计数",
        default=True,
    )

    show_mouse_events = bpy.props.BoolProperty(
        name="显示鼠标事件",
        default=True,
    )

    mouse_events_show_mode = bpy.props.EnumProperty(
        name="鼠标事件",
        items=[
            ('EVENT_HISTORY', "Event History", ""),
            ('HOLD_STATUS', "Hold Status", ""),
            ('EVENT_HISTORY_AND_HOLD_STATUS', "Event History + Hold Status", ""),
        ],
        default='HOLD_STATUS',
    )

    use_custom_mouse_image = bpy.props.BoolProperty(
        name="使用自定义鼠标图像",
        default=False,
        update=common.reload_custom_mouse_image,
    )

    custom_mouse_image_base = bpy.props.StringProperty(
        name="自定义鼠标图像（基础）",
        description="始终呈现的自定义鼠标图像",
        default="",
        update=common.reload_custom_mouse_image,
    )

    custom_mouse_image_overlay_left_mouse = bpy.props.StringProperty(
        name="自定义鼠标图像（覆盖 - 鼠标左键）",
        description="单击左键时呈现的自定义鼠标图像",
        default="",
        update=common.reload_custom_mouse_image,
    )

    custom_mouse_image_overlay_right_mouse = bpy.props.StringProperty(
        name="自定义鼠标图像（覆盖 - 鼠标右键）",
        description="单击右键时呈现的自定义鼠标图像",
        default="",
        update=common.reload_custom_mouse_image,
    )

    custom_mouse_image_overlay_middle_mouse = bpy.props.StringProperty(
        name="自定义鼠标图像（覆盖 - 鼠标中键）",
        description="单击中间按钮时呈现的自定义鼠标图像",
        default="",
        update=common.reload_custom_mouse_image,
    )

    use_custom_mouse_image_size = bpy.props.BoolProperty(
        name="使用自定义鼠标图像大小",
        description="使用自定义鼠标图像大小",
        default=False,
        update=update_custom_mouse_size,
    )

    custom_mouse_size = bpy.props.IntVectorProperty(
        name="自定义鼠标图像大小",
        description="自定义鼠标图像大小",
        default=(int(compat.get_user_preferences(bpy.context).ui_styles[0].widget.points*3),
                 int(compat.get_user_preferences(bpy.context).ui_styles[0].widget.points*3)),
        min=18,
        max=1000,
        size=2,
        subtype='XYZ',
    )

    show_last_operator = bpy.props.BoolProperty(
        name="显示最后一个运算符",
        default=False,
    )

    last_operator_show_mode = bpy.props.EnumProperty(
        name="最后一个运算符",
        items=[
            ('LABEL', "Label", ""),
            ('IDNAME', "ID Name", ""),
            ('LABEL_AND_IDNAME', "Label + ID Name", ""),
        ],
        default='LABEL_AND_IDNAME',
    )

    get_event_aggressively = bpy.props.BoolProperty(
        name="积极获取事件",
        description="""(Experimental) Get events which will be dropped by the
                       other modalhandlers. This may make blender unstable""",
        default=True,
    )

    auto_save = bpy.props.BoolProperty(
        name="自动保存",
        description="""(Experimental) Enable custom auto save while modal operator is running.
                       This may make blender unstable""",
        default=False,
    )

    output_debug_log = bpy.props.BoolProperty(
        name="输出调试日志",
        description="（调试）输出日志消息",
        default=False
    )

    display_draw_area = bpy.props.BoolProperty(
        name="显示绘图区",
        description="（调试）显示绘制区域",
        default=False
    )

    # for UI.
    def panel_space_type_items_fn(self, _):
        space_types = compat.get_all_space_types()
        items = []
        for i, (identifier, space) in enumerate(space_types.items()):
            space_name = space.bl_rna.name
            space_name = space_name.replace(" Space", "")
            space_name = space_name.replace("Space ", "")
            items.append((identifier, space_name, space_name, i))
        return items

    def ui_in_sidebar_update_fn(self, context):
        has_panel = hasattr(bpy.types, SK_PT_ScreencastKeys.bl_idname)
        if has_panel:
            try:
                bpy.utils.unregister_class(SK_PT_ScreencastKeys)
            except:
                pass

        if self.show_ui_in_sidebar:
            SK_PT_ScreencastKeys.bl_space_type = self.panel_space_type
            SK_PT_ScreencastKeys.bl_category = self.panel_category
            bpy.utils.register_class(SK_PT_ScreencastKeys)

    panel_space_type = bpy.props.EnumProperty(
        name="空间",
        description="显示 ScreencastKey 面板的空间",
        items=panel_space_type_items_fn,
        update=ui_in_sidebar_update_fn,
    )

    panel_category = bpy.props.StringProperty(
        name="类别",
        description="显示 ScreencastKey 面板的类别",
        default="Screencast Keys",
        update=ui_in_sidebar_update_fn,
    )

    enable_on_startup = bpy.props.BoolProperty(
        name="启动时启用",
        description="""Automatically enable Screencast Keys when blender is starting up""",
        default=False
    )

    show_ui_in_sidebar = bpy.props.BoolProperty(
        name="侧边栏",
        description="在侧边栏中显示 UI",
        default=True,
        update=ui_in_sidebar_update_fn,
    )

    def ui_in_overlay_update_fn(self, context):
        has_panel = hasattr(bpy.types, SK_PT_ScreencastKeys_Overlay.bl_idname)
        if has_panel:
            try:
                bpy.utils.unregister_class(SK_PT_ScreencastKeys_Overlay)
            except:
                pass
        if self.show_ui_in_overlay:
            bpy.utils.register_class(SK_PT_ScreencastKeys_Overlay)

    show_ui_in_overlay = bpy.props.BoolProperty(
        name="覆盖",
        description="在叠加层中显示 UI",
        default=False,
        update=ui_in_overlay_update_fn,
    )

    # for display event text alias
    enable_display_event_text_aliases = bpy.props.BoolProperty(
        name="启用显示事件文本别名",
        description="启用显示事件文本别名",
        default=False,
        update=ui_in_overlay_update_fn,
    )

    display_event_text_aliases_props = bpy.props.CollectionProperty(
        type=DisplayEventTextAliasProperties
    )

    # for add-on updater
    updater_branch_to_update = EnumProperty(
        name="branch",
        description="目标分支以更新附加组件",
        items=get_update_candidate_branches
    )

    def draw(self, context):
        layout = self.layout

        layout.row().prop(self, "category", expand=True)

        if self.category == 'CONFIG':
            layout.separator()

            layout.prop(self, "enable_on_startup")

            column = layout.column()
            split = column.split()
            col = split.column()
            col.prop(self, "color")
            col.separator()
            col.prop(self, "shadow")
            if self.shadow:
                col.prop(self, "shadow_color", text="")
            col.separator()
            col.prop(self, "background")
            if self.background:
                sp = compat.layout_split(col, factor=0.5)
                sp.prop(self, "background_mode", text="")
                sp = compat.layout_split(sp, factor=1.0)
                sp.prop(self, "background_color", text="")
                col.prop(self, "background_rounded_corner_radius", text="边角半径")
            col.separator()
            col.prop(self, "font_size")
            col.prop(self, "margin")
            col.prop(self, "line_thickness")

            col = split.column()
            col.prop(self, "origin")

            col.separator()
            col.prop(self, "align")
            col.separator()
            col.prop(self, "offset")
            col.separator()
            col.prop(self, "display_time")

            col = split.column()
            col.prop(self, "max_event_history")
            col.separator()
            col.prop(self, "repeat_count")
            col.separator()
            col.prop(self, "show_mouse_events")
            if self.show_mouse_events:
                col.prop(self, "mouse_events_show_mode")
            col.separator()
            col.prop(self, "show_last_operator")
            if self.show_last_operator:
                col.prop(self, "last_operator_show_mode")

            layout.prop(self, "use_custom_mouse_image")
            if show_mouse_hold_status(self):
                if self.use_custom_mouse_image:
                    row = layout.row()
                    r = row.row()
                    r.prop(self, "use_custom_mouse_image_size", text="使用图像大小")
                    r = row.row()
                    r.prop(self, "custom_mouse_size", text="尺寸")
                    r.enabled = not self.use_custom_mouse_image_size

                    column = layout.column()
                    split = column.split()

                    col = split.column()
                    col.label(text="基本:")
                    r = col.row(align=True)
                    r.prop(self, "custom_mouse_image_base", text="")
                    ops = r.operator(SK_OT_SelectCustomMouseImage.bl_idname, text="", icon='FILEBROWSER')
                    ops.target = 'BASE'
                    if common.CUSTOM_MOUSE_IMAGE_BASE_NAME in bpy.data.images:
                        image = bpy.data.images[common.CUSTOM_MOUSE_IMAGE_BASE_NAME]
                        col.template_icon(image.preview.icon_id, scale=2.0)

                    col = split.column()
                    col.label(text="叠加（左）")
                    r = col.row(align=True)
                    r.prop(self, "custom_mouse_image_overlay_left_mouse", text="")
                    ops = r.operator(SK_OT_SelectCustomMouseImage.bl_idname, text="", icon='FILEBROWSER')
                    ops.target = 'OVERLAY_LEFT_MOUSE'
                    if common.CUSTOM_MOUSE_IMAGE_OVERLAY_LEFT_MOUSE_NAME in bpy.data.images:
                        image = bpy.data.images[common.CUSTOM_MOUSE_IMAGE_OVERLAY_LEFT_MOUSE_NAME]
                        col.template_icon(image.preview.icon_id, scale=2.0)

                    col = split.column()
                    col.label(text="叠加（右）")
                    r = col.row(align=True)
                    r.prop(self, "custom_mouse_image_overlay_right_mouse", text="")
                    ops = r.operator(SK_OT_SelectCustomMouseImage.bl_idname, text="", icon='FILEBROWSER')
                    ops.target = 'OVERLAY_RIGHT_MOUSE'
                    if common.CUSTOM_MOUSE_IMAGE_OVERLAY_RIGHT_MOUSE_NAME in bpy.data.images:
                        image = bpy.data.images[common.CUSTOM_MOUSE_IMAGE_OVERLAY_RIGHT_MOUSE_NAME]
                        col.template_icon(image.preview.icon_id, scale=2.0)

                    col = split.column()
                    col.label(text="叠加（中）")
                    r = col.row(align=True)
                    r.prop(self, "custom_mouse_image_overlay_middle_mouse", text="")
                    ops = r.operator(SK_OT_SelectCustomMouseImage.bl_idname, text="", icon='FILEBROWSER')
                    ops.target = 'OVERLAY_MIDDLE_MOUSE'
                    if common.CUSTOM_MOUSE_IMAGE_OVERLAY_MIDDLE_MOUSE_NAME in bpy.data.images:
                        image = bpy.data.images[common.CUSTOM_MOUSE_IMAGE_OVERLAY_MIDDLE_MOUSE_NAME]
                        col.template_icon(image.preview.icon_id, scale=2.0)
                else:
                    column = layout.split(factor=0.5)
                    row = column.row()
                    row.prop(self, "mouse_size")

            # Panel location is only available in >= 2.80
            if compat.check_version(2, 80, 0) >= 0:
                layout.separator()

                layout.label(text="UI:")
                col = layout.column()
                col.prop(self, "show_ui_in_sidebar")

                if self.show_ui_in_sidebar:
                    col.label(text="面板位置:")
                    col.prop(self, "panel_space_type")
                    col.prop(self, "panel_category")

                col.separator()

                col.prop(self, "show_ui_in_overlay")

            layout.separator()

            layout.label(text="实验:")
            col = layout.column()
            col.prop(self, "get_event_aggressively")
            col.prop(self, "auto_save")

            layout.separator()

            layout.label(text="开发:")
            col = layout.column()
            col.prop(self, "output_debug_log")
            col.prop(self, "display_draw_area")

        elif self.category == 'DISPLAY_EVENT_TEXT_ALIAS':
            layout.separator()

            layout.prop(self, "enable_display_event_text_aliases")

            layout.separator()

            if self.enable_display_event_text_aliases:
                sp = compat.layout_split(layout, factor=0.33)
                col = sp.column()
                col.label(text="事件编号")
                sp = compat.layout_split(sp, factor=0.5)
                col = sp.column()
                col.label(text="默认文字")
                sp = compat.layout_split(sp, factor=1.0)
                col = sp.column()
                col.label(text="别名文字")

                layout.separator()

                for d in self.display_event_text_aliases_props:
                    sp = compat.layout_split(layout, factor=0.33)
                    col = sp.column()
                    col.label(text=d.event_id)
                    sp = compat.layout_split(sp, factor=0.5)
                    col = sp.column()
                    col.label(text=d.default_text)
                    sp = compat.layout_split(sp, factor=1.0)
                    col = sp.column()
                    col.prop(d, "alias_text", text="")

        elif self.category == 'UPDATE':
            updater = AddonUpdaterManager.get_instance()

            layout.separator()

            if not updater.candidate_checked():
                col = layout.column()
                col.scale_y = 2
                row = col.row()
                row.operator(SK_OT_CheckAddonUpdate.bl_idname,
                             text="Check 'Screencast Keys' add-on update",
                             icon='FILE_REFRESH')
            else:
                row = layout.row(align=True)
                row.scale_y = 2
                col = row.column()
                col.operator(SK_OT_CheckAddonUpdate.bl_idname,
                             text="Check 'Screencast Keys' add-on update",
                             icon='FILE_REFRESH')
                col = row.column()
                if updater.latest_version() != "":
                    col.enabled = True
                    ops = col.operator(
                        SK_OT_UpdateAddon.bl_idname,
                        text="Update to the latest release version (version: {})"
                            .format(updater.latest_version()),
                        icon='TRIA_DOWN_BAR')
                    ops.branch_name = updater.latest_version()
                else:
                    col.enabled = False
                    col.operator(SK_OT_UpdateAddon.bl_idname,
                                 text="没有可用的更新.")

                layout.separator()
                layout.label(text="手动更新:")
                row = layout.row(align=True)
                row.prop(self, "updater_branch_to_update", text="目标")
                ops = row.operator(
                    SK_OT_UpdateAddon.bl_idname, text="更新",
                    icon='TRIA_DOWN_BAR')
                ops.branch_name = self.updater_branch_to_update

                layout.separator()
                if updater.has_error():
                    box = layout.box()
                    box.label(text=updater.error(), icon='CANCEL')
                elif updater.has_info():
                    box = layout.box()
                    box.label(text=updater.info(), icon='ERROR')
